<?php
namespace Modules\Review\Models;

use App\BaseModel;
use App\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\SoftDeletes;
use Modules\Review\Models\ReviewMeta;

class Review extends BaseModel
{
    use SoftDeletes;
    protected $table    = 'bc_review';
    protected $fillable = [
        'object_id',
        'object_model',
        'title',
        'content',
        'rate_number',
        'author_ip',
        'status',
        'object_author_id',
        'author_id',
    ];

    public static function getDisplayTextScoreByLever($lever)
    {
        switch ($lever) {
            case 5:
                return __("Excellent");
                break;
            case 4:
                return __("Very Good");
                break;
            case 3:
                return __("Average");
                break;
            case 2:
                return __("Poor");
                break;
            case 1:
            case 0:
                return __("Terrible");
                break;
            default:
                return __("Not rated");
                break;
        }
    }

    public function getService()
    {
        $allServices = get_reviewable_services();
        $module = $allServices[$this->object_model];
        return $this->belongsTo($module, 'object_id');
    }

    public function getReviewMeta()
    {
        return ReviewMeta::where("review_id", $this->id)->get();
    }

    public function getReviewMetaPicture()
    {
        return ReviewMeta::where("review_id", $this->id)->where('name', 'upload_picture')->first();
    }

    public static function countReviewByStatus($status = false)
    {
        $count = parent::query();
        if (!empty($status)) {
            $count->where("status", $status);
        }
        return $count->count("id");
    }

    public static function countReviewByServiceID($service_id = false, $user_id = false, $status = false,$service_type = '')
    {
        if (empty($service_id))
            return false;
        $count = parent::where("object_id", $service_id);
        if (!empty($status)) {
            $count->where("status", $status);
        }

        if($service_type){
            $count->where('object_model',$service_type);
        }
        if (!empty($user_id)) {
            $count->where("create_user", $user_id);
        }
        return $count->count("id");
    }

    public function save(array $options = [])
    {
        $check = parent::save($options); // TODO: Change the autogenerated stub
        if ($check) {
            Cache::forget("review_" . $this->object_model . "_" . $this->object_id);
        }
        return $check;
    }

    public function addMeta($key, $val, $multiple = false)
    {
        if (is_object($val) or is_array($val))
            $val = json_encode($val);
        if ($multiple) {
            $reviewMeta = new ReviewMeta([
                "object_id"    => $this->object_id,
                "object_model" => $this->object_model,
                "name"         => $key,
                "val"          => $val,
                'review_id'=>$this->id
            ]);
            return $reviewMeta->save();

        } else {
            $old = ReviewMeta::query()->where([
                    'review_id' => $this->id,
                    'name'       => $key
                ])->first();

            if ($old) {
                $old->val = $val;
                return $old->save();

            } else {
                $reviewMeta = new ReviewMeta([
                    "object_id"    => $this->object_id,
                    "object_model" => $this->object_model,
                    "name"         => $key,
                    "val"          => $val,
                    'review_id'=>$this->id
                ]);
                return $reviewMeta->save();
            }
        }
    }

    public static function getTotalViewAndRateAvg($objectId, $objectModel)
    {
        $list_score = [
            'score_total'  => 0,
            'total_review' => 0,
        ];
        if (empty($objectId) || empty($objectModel)) {
            return $list_score;
        }

        $list_score = Cache::rememberForever('review_'.$objectModel.'_' . $objectId, function () use ($objectId, $objectModel) {
            $dataReview = self::selectRaw(" AVG(rate_number) as score_total , COUNT(id) as total_review ")->where('object_id', $objectId)->where('object_model', $objectModel)->where("status", "approved")->first();
            $score_total = !empty($dataReview->score_total) ? number_format($dataReview->score_total, 1) : 0;
            return [
                'score_total'  => $score_total,
                'total_review' => !empty($dataReview->total_review) ? $dataReview->total_review : 0,
            ];
        });
        return $list_score;
    }

    public function author()
    {
        return $this->belongsTo(User::class, "author_id")->select(['id','name','first_name','last_name','avatar_id'])->withDefault();
    }
}
