<?php


namespace Modules\Order\Models;


use App\BaseModel;
use App\Traits\HasMeta;
use App\User;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Notification;
use Modules\Order\Events\OrderItemStatusUpdated;
use Modules\Order\Notifications\OrderItemNotification;
use Modules\Order\Traits\HasDownloadable;

class OrderItem extends BaseModel
{

    use SoftDeletes;
    use HasMeta;
    use HasDownloadable;

    protected $table = 'core_order_items';

    protected $meta_parent_key = 'order_item_id';
    protected $metaClass = OrderItemMeta::class;

    protected $casts = [
        'meta'=>'array'
    ];

    public function order(){
        return $this->belongsTo(Order::class,'order_id');
    }
    public function vendor(){
        return $this->belongsTo(User::class,'vendor_id');
    }

    public function subtotal(): Attribute
    {
        return Attribute::make(
            get:function($value){
                return $this->price * $this->qty + $this->extra_price_total;

            }
        );
    }

    public function extraPriceTotal(): Attribute
    {
        return Attribute::make(
            get:function($value){
                $t = 0;
                if(!empty($this->meta['extra_prices']))
                {
                    foreach ($this->meta['extra_prices'] as $extra_price){
                        $t += (float)($extra_price['price']);
                    }
                }
                return $t;
            }
        );
    }

    public static function search($filters = [])
    {
        $query = parent::query()->select(['core_order_items.*','core_orders.customer_id']);
        $query->join('core_orders','core_orders.id','=','core_order_items.order_id');

        if(!empty($filters['vendor_id']))
        {
            $query->where('vendor_id',$filters['vendor_id']);
        }
        if(!empty($filters['status']))
        {
            $query->where('core_order_items.status',$filters['status']);
        }
        if(!empty($filters['s']))
        {
            $query->where(function($query) use ($filters){
                return $query->where('object_id',$filters['s'])->orWhere('core_order_items.id',$filters['s']);
            });
        }
        return $query;
    }


    public function product() : Attribute{
        return Attribute::make(
            get: function () {
                $keys = get_services();
                if(!empty($keys[$this->object_model])){
                    $model = app()->make($keys[$this->object_model]);
                    return $model->find($this->object_id);
                }
                return null;
            }
        );
    }

    public function save(array $options = [])
    {

        if(!$this->locale){
            $this->locale = app()->getLocale();
        }
        if(!$this->order_date){
            $this->order_date = Carbon::now();
        }
        return parent::save($options); // TODO: Change the autogenerated stub
    }

    public function calculateCommission($saveMeta = true){
        if(is_vendor_enable()){
            $commission = [];
            $commission['amount'] = setting_item('vendor_commission_amount', 0);
            $commission['type'] = setting_item('vendor_commission_type', 'percent');
            $vendor = $this->vendor;
            if($vendor){
                if($vendor->vendor_commission_type){
                    $commission['type'] = $vendor->vendor_commission_type;
                }
                if($vendor->vendor_commission_amount){
                    $commission['amount'] = $vendor->vendor_commission_amount;
                }
                if($commission['type'] == 'disable'){
                    // No Commission for Admin
                    return;
                }
            }
            if ($commission['type'] == 'percent') {
                $this->commission_amount = (float)($this->subtotal / 100) * $commission['amount'];
            } else {
                $this->commission_amount = (float)min($this->subtotal,$commission['amount']);
            }
            if($saveMeta){
                $this->addMeta('commission',$commission);
            }
        }
    }

    public function calculateTotal(){
        $this->setAttribute('subtotal',$this->subtotal);
    }


    public function updatePrice(){
        if($this->product){
            if($this->variation_id){
                $this->price = ProductVariation::find($this->variation_id)->sale_price ?? 0;
            }else{
                $this->price = $this->product->getBuyablePrice();
            }
        }
        $this->save();
    }



    public function getEditableStatues(){
        switch ($this->status){
            case Order::PROCESSING :
            case Order::FAILED :
                return [Order::CANCELLED];
                break;
            case Order::ON_HOLD :
                return [Order::PROCESSING,Order::CANCELLED];
                break;
            default:
                return [];
        }
    }

    /**
     * Change Order item Status, Add note, Dispatch event
     *
     * @param $status
     */
    public function updateStatus($status){

        if($status === $this->status){
            return;
        }

        $old_status = $this->status;
        $this->status = $status;

        $this->save();


        $this->order->addNote(OrderNote::ITEM_STATUS_CHANGED,__("Order Item :id status changed from :old_status to :new_status",['id'=>$this->id,'old_status'=>$old_status,'new_status'=>$status]),['old_status'=>$old_status,'new_status'=>$status]);

        OrderItemStatusUpdated::dispatch($this,$old_status,$status);
    }

    public function reduceStock()
    {
        if(empty($this->reduced_stock)){
            $model = $this->product;
            if(!empty($model) and $model instanceof  Product){
                if($model->check_manage_stock()){
                    $model->quantity -= $this->qty;
                    $model->sale_count += $this->qty;
                    if($model->quantity <=0){
                        $model->quantity = 0 ;
                        $model->stock_status ='out';
                    }
                    $model->save();
                } elseif (!empty($this->variation_id)){
                    $variation = $model->variations()->where('id',$this->variation_id)->first();
                    if(!empty($variation) and $variation->check_manage_stock()){
                        $variation->quantity -= $this->qty;
                        $variation->sale_count += $this->qty;
                        if($variation->quantity <=0){
                            $variation->quantity = 0 ;
                            $variation->stock_status ='out';
                        }
                        $variation->save();
                    }
                }
            }
            $this->reduced_stock = $this->qty;
            $this->save();
        }

    }
    public function returnStock(){
        if(!empty($this->reduced_stock)){
            $model = $this->product;
            if(!empty($model) and $model instanceof  Product){
                if($model->check_manage_stock()){
                    $model->quantity += $this->reduced_stock;
                    $model->sale_count -= $this->qty;
                    $model->sale_count = max(0,$model->sale_count);
                    if($model->quantity<=0){
                        $model->stock_status ='out';
                    }
                    $model->save();
                } elseif (!empty($this->variation_id)) {
                    $variation = $model->variations()->where('id',$this->variation_id)->first();
                    if(!empty($variation) and $variation->check_manage_stock()){
                        $variation->quantity += $this->reduced_stock;
                        $variation->sale_count -= $this->qty;
                        $variation->sale_count = max(0,$variation->sale_count);
                        if($variation->quantity<=0){
                            $variation->stock_status ='out';
                        }
                        $variation->save();
                    }
                }
            }
            $this->reduced_stock = null;
            $this->save();
        }
    }

    public function sendOrderNotifications($type){

        // Send Email
        if(setting_item('email_c_'.$type.'_enable')) {
            $customer = $this->order->customer ?? null;
            if($customer){
                $customer->notify(new OrderItemNotification($this,$type));
            }
        }
        if(setting_item('email_v_'.$type.'_enable') and is_vendor_enable()) {
            if($vendor = $this->author){
                $vendor->notify(new OrderItemNotification($this,$type,'vendor',$vendor));
            }
        }

        if(setting_item('email_a_'.$type.'_enable') and $address = setting_item('admin_email')) {
            Notification::route('mail',$address)->notify(new OrderItemNotification($this,$type,'admin'));
        }
        return true;
    }

    public function variation()
    {
        return $this->belongsTo(ProductVariation::class, 'variation_id');
    }

    public function getDetailUrl()
    {
        if ($this->product) {
            return $this->product->getDetailUrl();
        }
        return '';
    }

    public function syncTotal(){

        // NOTE: Only calculate data, do not save to database

        // Save will be done in after save event

        $this->subtotal = $this->price * $this->qty + $this->extra_price_total;
    }



    // Get item info
    public function getItemName(){
        return $this->title ?? $this->product->title  ?? '';
    }

    public function getItemPrice(){
        return $this->price;
    }

    public function getItemQuantity(){
        return $this->qty;
    }

    public function getItemSubtotal(){
        return $this->subtotal;
    }

    public function getItemTotal(){
        return $this->subtotal;
    }

    public function getItemImageUrl($size = 'thumb'){
        if($this->product){
            return $this->product->image_id ? get_image_url($this->product->image_id, $size) : '';
        }
        return '';
    }
    
}
