<?php
/**
 * Date: 10/10/2023
 */
namespace Modules\Course\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Modules\AdminController;
use Modules\Course\Models\Course;
use Modules\Course\Models\CourseLevel;
use Modules\Course\Models\CourseTerm;
use Modules\Course\Models\CourseTranslation;
use Modules\Core\Models\Attributes;
use Modules\Course\Models\CourseCategory;
class CourseController extends AdminController
{
    protected $course;
    protected $course_translation;
    protected $course_term;
    protected $attributes;
    protected $location;
    protected $courseCategoryClass;
    protected $courseLevel;
    public function __construct()
    {
        $this->setActiveMenu(route('course.admin.index'));
        $this->course = Course::class;
        $this->course_translation = CourseTranslation::class;
        $this->course_term = CourseTerm::class;
        $this->attributes = Attributes::class;
        $this->courseCategoryClass = CourseCategory::class;
        $this->courseLevel = CourseLevel::class;
    }

    public function callAction($method, $parameters)
    {
        if(!Course::isEnable())
        {
            return redirect('/');
        }
        return parent::callAction($method, $parameters); // TODO: Change the autogenerated stub
    }

    public function index(Request $request)
    {

        $this->checkPermission('course_view');
        $query = $this->course::query() ;
        $query->orderBy('id', 'desc');
        if (!empty($s = $request->input('s'))) {
            $query->where('title', 'LIKE', '%' . $s . '%');
            $query->orderBy('title', 'asc');
        }

        if ($this->hasPermission('course_manage_others')) {
            if (!empty($author = $request->input('author_id'))) {
                $query->where('author_id', $author);
            }
        } else {
            $query->where('author_id', Auth::id());
        }
        $data = [
            'rows'               => $query->with(['author','category'])->withCount(['students'])->paginate(20),
            'course_manage_others' => $this->hasPermission('course_manage_others'),
            'breadcrumbs'        => [
                [
                    'name' => __('Courses'),
                    'url'  => route('course.admin.index')
                ],
                [
                    'name'  => __('All'),
                    'class' => 'active'
                ],
            ],
            'page_title'=>__("Course Management")
        ];
        return view('Course::admin.index', $data);
    }

    public function create(Request $request)
    {
        $this->checkPermission('course_create');
        $row = new $this->course();
        $row->fill([
            'status' => 'publish'
        ]);
        $data = [
            'row'            => $row,
            'translation'    => new $this->course_translation(),
            'attributes'     => $this->attributes::where('service', 'course')->get(),
            'category_tree'=> $this->courseCategoryClass::where('status', 'publish')->get()->toTree(),
            'levels'=> $this->courseLevel::where('status', 'publish')->get(),
            'breadcrumbs'    => [
                [
                    'name' => __('Courses'),
                    'url'  => 'course.admin.index'
                ],
                [
                    'name'  => __('Add Course'),
                    'class' => 'active'
                ],
            ],
            'page_title'     => __("Add new Course")
        ];
        return view('Course::admin.detail', $data);
    }

    public function edit(Request $request, $id)
    {
        $this->checkPermission('course_update');
        $row = $this->course::find($id);
        if (empty($row)) {
            return redirect(route('course.admin.index'));
        }
        $translation = $row->translate($request->query('lang'));
        if (!$this->hasPermission('course_manage_others')) {
            if ($row->create_user != Auth::id()) {
                return redirect(route('course.admin.index'));
            }
        }
        $data = [
            'row'            => $row,
            'translation'    => $translation,
            "selected_terms" => $row->terms->pluck('term_id'),
            'attributes'     => $this->attributes::where('service', 'course')->get(),
            'category_tree'=>$this->courseCategoryClass::where('status', 'publish')->get()->toTree(),
            'levels' => $this->courseLevel::where('status', 'publish')->get(),
            'enable_multi_lang'=>true,
            'breadcrumbs'    => [
                [
                    'name' => __('Courses'),
                    'url'  => route('course.admin.index')
                ],
                [
                    'name'  => __('Edit Course'),
                    'class' => 'active'
                ],
            ],
            'page_title'=>__("Edit: :name",['name'=>$row->title])
        ];
        return view('Course::admin.detail', $data);
    }

    public function store( Request $request, $id ){

        if($id>0){
            $this->checkPermission('course_update');
            $row = $this->course::find($id);
            if (empty($row)) {
                return redirect(route('course.admin.index'));
            }

            if($row->create_user != Auth::id() and !$this->hasPermission('course_manage_others'))
            {
                return redirect(route('course.admin.index'));
            }
        }else{
            $this->checkPermission('course_create');
            $row = new $this->course();
            $row->status = "publish";
        }
        $dataKeys = [
            'title',
            'content',
            'slug',
            'status',
            'image_id',
            'price',
            'original_price',
            'is_featured',
            'cat_id',
            'level_id',
            'author_id',
            'preview_url',
            'duration',
            'language'
        ];
        if($this->hasPermission('course_manage_others')){
            $dataKeys[] = 'author_id';
        }
        $row->fillByAttr($dataKeys,$request->input());

        $res = $row->saveOriginOrTranslation($request->input('lang'),true);

        if ($res) {
            if(!$request->input('lang') or is_default_lang($request->input('lang'))) {
                $this->saveTerms($row, $request);
            }

            if($id > 0 ){
                return back()->with('success',  __('Course updated') );
            }else{
                return redirect(route('course.admin.edit',$row->id))->with('success', __('Course created') );
            }
        }
    }

    public function saveTerms($row, $request)
    {
        $this->checkPermission('course_manage_attributes');
        if (empty($request->input('terms'))) {
            $this->course_term::where('target_id', $row->id)->delete();
        } else {
            $term_ids = $request->input('terms');
            foreach ($term_ids as $term_id) {
                $this->course_term::firstOrCreate([
                    'term_id' => $term_id,
                    'target_id' => $row->id
                ]);
            }
            $this->course_term::where('target_id', $row->id)->whereNotIn('term_id', $term_ids)->delete();
        }
    }

    public function bulkEdit(Request $request)
    {

        $ids = $request->input('ids');
        $action = $request->input('action');
        if (empty($ids) or !is_array($ids)) {
            return redirect()->back()->with('error', __('No items selected!'));
        }
        if (empty($action)) {
            return redirect()->back()->with('error', __('Please select an action!'));
        }

        switch ($action){
            case "delete":
                foreach ($ids as $id) {
                    $query = $this->course::where("id", $id);
                    if (!$this->hasPermission('course_manage_others')) {
                        $query->where("create_user", Auth::id());
                        $this->checkPermission('course_delete');
                    }
                    $query->first();
                    if(!empty($query)){
                        $query->delete();
                    }
                }
                return redirect()->back()->with('success', __('Deleted success!'));
                break;
            case "clone":
                $this->checkPermission('course_create');
                foreach ($ids as $id) {
                    (new $this->course())->saveCloneByID($id);
                }
                return redirect()->back()->with('success', __('Clone success!'));
                break;
            default:
                // Change status
                foreach ($ids as $id) {
                    $query = $this->course::where("id", $id);
                    if (!$this->hasPermission('course_manage_others')) {
                        $query->where("create_user", Auth::id());
                        $this->checkPermission('course_update');
                    }
                    $query->update(['status' => $action]);
                }
                return redirect()->back()->with('success', __('Update success!'));
                break;
        }


    }
}
